/* *********************************************************************
   *                                                                   *
   *  NAME     : Mariusz Zaczek          PROGRAM : MP5                 *
   *  NET ID   : zaczek                  COURSE  : CS318 - Fall 1999   *
   *  DUE DATE : Nov. 19,1999                                          *
   *                                                                   *
   *  PURPOSE  : The purpose of this MP is to draw a Fractal Mountain  *
   *                                                                   *
   *  INPUTS   : MOUSE & KEYBOARD INPUTS:                              *
   *             Left Mouse Button  - Click/Hold to Rotate Camera      *
   *                                   Azimuth/Elevation               *
   *             Right Mouse Button - Click/Hold to Zoom (Up/Down)     *
   *                                   or Twist (Left/Right)           *
   *             1 to 6             - Set Level of Detail/Subdivision  *
   *             q or Q             - used to quit the program         *
   *                                                                   *
   ********************************************************************* */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <GL/glut.h>

/* Global variables for screen width and height*/
GLint  WinWidth = 600;
GLint WinHeight = 600;

/* Point structure */
typedef struct
{
  GLfloat x,y,z; 
  GLint   c;
} Vertex;


Vertex Vertices[65][65];    /* the 2D grid of the landscape/mountain    */
GLint   Calc = 1,           /* boolean for calculating of all vertices  */
        Azimuth = 30,       /* Azimuth, Elevation, and Twist are used   */
        Elevation = 60,     /*    to compute the camera position when   */
        Twist = 0,         
        S = 1;              /* the number of subdivisions               */

GLfloat Focal = -11.0,      /* Distance from origin to camera           */
        offset,             /* the offset value from RandomGaussian     */
        Mean = 0.0,         /* the mean for the landscape               */
        MeanR,              /* the mean for the mountain                */
        StdDev,             /* the standard deviation for the landscape */
        StdDevR,            /* the standard deviation for the mountain  */
        D = 2.8,            /* the fractal dimension                    */ 
        H,                  /* H = 3 - D                                */
        A,                  /* A = 2(1 - D)                             */
        Scale1 = 0.5,       /* the scale factor for landscape StdDev    */
        Scale2 = 0.5; 


/***********************************************************************/
/*  RandomGaussian returns a random number from a Gaussian             */
/*  distribution with the given "mean" and "stddev".  This algorithm   */
/*  was adapted from "Numerical Recipies in C" in the section on       */
/*  "Normal Deviates".  Include <stdlib.h> for RAND_MAX.               */
/***********************************************************************/
float RandomGaussian(float mean, float stddev)
{
  float factor;
  float radiussq;
  float value1, value2;

  do
  {
    value1 = 2.0 * (float)rand() / RAND_MAX - 1.0;
    value2 = 2.0 * (float)rand() / RAND_MAX - 1.0;
    radiussq = value1 * value1 + value2 * value2;
  } while ((radiussq >= 1.0) || ((radiussq - 0.000001) < 0.0));

  factor = sqrt(-2.0 * log(radiussq) / radiussq);

  return value1 * factor * stddev + mean;
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    reshape                                      +
   +                                                            +
   +  Purpose: This function allows the screen to be resized    +
   +            but the line draw still appears in its original +
   +            position and length.                            +
   +                                                            +
   +  Inputs: w and h - width and height of window              +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void reshape (int w, int h)
{
  glViewport(0, 0, (GLsizei) w, (GLsizei) h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluOrtho2D(0.0, (GLdouble) w,(GLdouble) h, 0.0);
}

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    SetCameraPosition                            +
   +                                                            +
   +  Purpose: This function sets the postion of the camera     +
   +                                                            +
   +  Inputs: none                                              +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void SetCameraPosition(void)
{
  /* Set the viewing transformation */
  glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(15.0,1.0,1.0,100.0);
    glTranslated(0.0,0.0,Focal);       /* Move camera down the negative z-axis */
    glRotated(-Twist,0.0,0.0,1.0);     /* Do a polar projection for camera */
    glRotated(-Elevation,1.0,0.0,0.0);
    glRotated(Azimuth,0.0,0.0,1.0);
    glTranslated(0.0,0.0,-0.8);        /* Center camera about mountain */
  glMatrixMode(GL_MODELVIEW);
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    init                                         +
   +                                                            +
   +  Purpose: This function clears the colors and initializes  +
   +            the shade model.                                +
   +                                                            +
   +  Inputs: none                                              +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */   
void init(void)
{
  GLfloat  light_ambient[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat  light_diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat light_specular[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat light_position[] = { 3.0, 3.0, 3.0, 0.0 };
   
  GLfloat   mat_ambient[] = { 0.1, 0.4, 0.1, 1.0 };
  GLfloat   mat_diffuse[] = { 0.1, 0.4, 0.1, 1.0 };
  GLfloat  mat_specular[] = { 0.1, 0.4, 0.1, 1.0 };
  GLfloat mat_shininess[] = { 2.0 };

  glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
  glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);

  glLightfv(GL_LIGHT0, GL_AMBIENT, light_ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse); 
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT0, GL_POSITION, light_position);

  glClearColor(0.0, 0.0, 0.0, 0.0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

  SetCameraPosition();

  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glDepthFunc(GL_LEQUAL);

  glEnable(GL_DEPTH_TEST);
  glShadeModel(GL_FLAT);
  glEnable(GL_NORMALIZE);   
}

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    sqr                                          +
   +                                                            +
   +  Purpose: This function returns the square of a number     +
   +                                                            +
   +  Inputs: x (float) - number to be squared                  +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
float sqr(float x)
{
  return x * x;
}

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    InitVertices                                 +
   +                                                            +
   +  Purpose: Initialize all the vertices of the 65x65 array   +
   +           to zero except for the corner vertices.          +
   +                                                            +
   +  Inputs: none                                              +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void InitVertices(void)
{
  GLint i = 0, j = 0;
   
  /* Set all vertices to zero and set "SET" flag to zero */
  for (i = 0; i < 65; i++)
    for (j = 0; j < 65; j++)
    {  
      Vertices[i][j].x = Vertices[i][j].y = Vertices[i][j].z = 0.0;
      Vertices[i][j].c = 0;
    }
 
  /* Set corner vertices to appropriate coordinates */
  Vertices[0][0].x = Vertices[64][0].x = Vertices[64][0].y
                   = Vertices[64][64].y = -1.0;
  Vertices[0][0].y = Vertices[0][64].x = Vertices[0][64].y
                   = Vertices[64][64].x = 1.0;
  Vertices[0][0].z = Vertices[0][64].z = Vertices[64][0].z
                   = Vertices[64][64].z = 0.0;
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    DrawSquare                                   +
   +                                                            +
   +  Purpose: This function calculates the normals and draws   +
   +           the triangular planes.                           +
   +                                                            +
   +  Inputs: a,b,c,d (Vertex) - corner of new square           +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void DrawSquare(Vertex a, Vertex b, Vertex c, Vertex d)
{
  GLfloat v[] = { 0.0, 0.0, 0.0 };  /* the normal vector */
  Vertex X, Y;                      /* the X & Y vectors */

  /* Draw first Triangle Plane */ 
  /* Calculating X and Y vectors for cross product */
  X.x = a.x - b.x;     X.y = a.y - b.y;     X.z = a.z - b.z;
  Y.x = c.x - b.x;     Y.y = c.y - b.y;     Y.z = c.z - b.z;
   
  /* v is the cross product of X x Y */
  v[0] = (X.x * Y.z - Y.y * X.z);
  v[1] = (X.z * Y.x - Y.z * X.x);
  v[2] = (X.x * Y.y - Y.x * X.y);

  glBegin(GL_TRIANGLES);
     glNormal3fv(v);
     glVertex3f(a.x, a.y, a.z);
     glNormal3fv(v);
     glVertex3f(b.x, b.y, b.z);
     glNormal3fv(v);
     glVertex3f(c.x, c.y, c.z);
  glEnd();
      
  /* Draw second Triangle Plane */
  /* Calculating X and Y vectors for cross product */
  X.x = c.x - d.x;     X.y = c.y - d.y;     X.z = c.z - d.z;
  Y.x = a.x - d.x;     Y.y = a.y - d.y;     Y.z = a.z - d.z;

  /* v is the cross product of X x Y */
  v[0] = (X.x * Y.z - Y.y * X.z);
  v[1] = (X.z * Y.x - Y.z * X.x);
  v[2] = (X.x * Y.y - Y.x * X.y);
                   
  glBegin(GL_TRIANGLES);
     glNormal3fv(v);
     glVertex3f(c.x, c.y, c.z);
     glNormal3fv(v);
     glVertex3f(d.x, d.y, d.z);
     glNormal3fv(v);
     glVertex3f(a.x, a.y, a.z);
  glEnd();

  glFlush();         
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    DivideSquare                                 +
   +                                                            +
   +  Purpose: This is an recursive function which for each     +
   +           subdivision, divides the given square into 4     +
   +           equal squares.                                   +
   +                                                            +
   +  Inputs: a,b,c,d (Vertex) - corners of square              +
   +          i,j,I,J (GLint)  - Inidices of the Vertex array   +
   +                             corresponding to the corners   +
   +          S       (GLint)  - # of subdivisions              +
   +                                                            +
   +                                                            +
   +           y-axis                                           +
   +            ^                                               +
   +    (a)     |     (b)                                       +
   +      +-----o-----+                                         +
   +      |     |     |                                         +
   +      |     |     |                                         +
   +      o     +-----o-----> x-axis                            +
   +      |           |                                         + 
   +      |           |                                         +
   +      +-----o-----+                                         +
   +    (d)           (c)                                       +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void DivideSquare(Vertex a, Vertex b, Vertex c, Vertex d,
                   GLint i,  GLint j,  GLint I,  GLint J, GLint S)
{
  if (S > 0)
  {  
    if (Calc == 1)
    {
      H = 3.0 - D;
      StdDev = Scale1 * (float)pow((fabs(a.x - b.x)),(2.0 * H));
      offset = RandomGaussian(Mean,StdDev);

      /* Calculate coordinates for the TOP midpoint. */
      if (Vertices[i][(j+J)/2].c == 0)
      {
        Vertices[i][(j+J)/2].x = (a.x + b.x) / 2.0;
        Vertices[i][(j+J)/2].y = (a.y + b.y) / 2.0;
        Vertices[i][(j+J)/2].z = ((a.z + b.z) / 2.0) + offset;   

        A = 2.0 * ( 1 - (float)sqrt((float)sqr(Vertices[i][(j+J)/2].x)
                      + (float)sqr(Vertices[i][(j+J)/2].y)));
        if (A < 0)   
          A = 0;

        MeanR = A - Vertices[i][(j+J)/2].z;
        StdDevR = Scale2 * (fabs(MeanR));
        Vertices[i][(j+J)/2].z += RandomGaussian(MeanR,StdDevR);
         
        /* Assign a flag that this value has been calculated. */
        Vertices[i][(j+J)/2].c = 1;
      } 

      /* Calculate coordinates for the RIGHT midpoint */
      if (Vertices[(i+I)/2][J].c == 0)
      {
        Vertices[(i+I)/2][J].x = (b.x + c.x) / 2.0;
        Vertices[(i+I)/2][J].y = (b.y + c.y) / 2.0;
        Vertices[(i+I)/2][J].z = ((b.z + c.z) / 2.0) + offset;   

        A = 2.0 * ( 1 - (float)sqrt((float)sqr(Vertices[(i+I)/2][J].x)
                      + (float)sqr(Vertices[(i+I)/2][J].y)));
        if (A < 0)
          A = 0;

        MeanR = A - Vertices[(i+I)/2][J].z;
        StdDevR = Scale2 * (fabs(MeanR));
        Vertices[(i+I)/2][J].z += RandomGaussian(MeanR,StdDevR);
         
        /* Assign a flag that this value has been calculated. */
        Vertices[(i+I)/2][J].c = 1;
      }

      /* Calculate coordinates for the BOTTOM midpoint */
      if (Vertices[I][(j+J)/2].c == 0)
      {
        Vertices[I][(j+J)/2].x = (c.x + d.x) / 2.0;
        Vertices[I][(j+J)/2].y = (c.y + d.y) / 2.0;
        Vertices[I][(j+J)/2].z = ((c.z + d.z) / 2.0) + offset;     

        A = 2.0 * ( 1 - (float)sqrt((float)sqr(Vertices[I][(j+J)/2].x)
                      + (float)sqr(Vertices[I][(j+J)/2].y)));
        if (A < 0)
          A = 0;

        MeanR = A - Vertices[I][(j+J)/2].z;
        StdDevR = Scale2 * (fabs(MeanR));
        Vertices[I][(j+J)/2].z += RandomGaussian(MeanR,StdDevR);
         
        /* Assign a flag that this value has been calculated. */
        Vertices[I][(j+J)/2].c = 1;
      }

      /* Calculate coordinates for the LEFT midpoint */
      if (Vertices[(i+I)/2][j].c == 0)
      {
        Vertices[(i+I)/2][j].x = (d.x + a.x) / 2.0;
        Vertices[(i+I)/2][j].y = (d.y + a.y) / 2.0;
        Vertices[(i+I)/2][j].z = ((d.z + a.z) / 2.0) + offset;   

        A = 2.0 * ( 1 - (float)sqrt((float)sqr(Vertices[(i+I)/2][j].x)
                      + (float)sqr(Vertices[(i+I)/2][j].y)));
        if (A < 0)
          A = 0;

        MeanR = A - Vertices[(i+I)/2][j].z;
        StdDevR = Scale2 * (fabs(MeanR));
        Vertices[(i+I)/2][j].z += RandomGaussian(MeanR,StdDevR);
            
        /* Assign a flag that this value has been calculated. */
        Vertices[(i+I)/2][j].c = 1;
      }

      /* Calculate coordinates for the midpoint of the square */
      if (Vertices[(i+I)/2][(j+J)/2].c == 0)
      {
        Vertices[(i+I)/2][(j+J)/2].x = (a.x + b.x + c.x + d.x) / 4.0;
        Vertices[(i+I)/2][(j+J)/2].y = (a.y + b.y + c.y + d.y) / 4.0;
        Vertices[(i+I)/2][(j+J)/2].z = ((a.z + b.z + c.z + d.z) / 4.0) + offset;

        A = 2.0 * ( 1 - (float)sqrt((float)sqr(Vertices[(i+I)/2][(j+J)/2].x)
                      + (float)sqr(Vertices[(i+I)/2][(j+J)/2].y)));
        if (A < 0)
          A = 0;

        MeanR = A - Vertices[(i+I)/2][(j+J)/2].z;
        StdDevR = Scale2 * (fabs(MeanR));
        Vertices[(i+I)/2][(j+J)/2].z += RandomGaussian(MeanR,StdDevR);

        /* Assign a flag that this value has been calculated. */
        Vertices[(i+I)/2][(j+J)/2].c = 1;
      }
   }     /* End of ..(Calc == 1) */

   /* If the "current" subdivision number is 1 then draw the square */
   if (S == 1)
   {     
     DrawSquare(a, Vertices[i][(j+J)/2], Vertices[(i+I)/2][(j+J)/2],
                Vertices[(i+I)/2][j]);
     DrawSquare(Vertices[i][(j+J)/2], b, Vertices[(i+I)/2][J],
                Vertices[(i+I)/2][(j+J)/2]);
     DrawSquare(Vertices[(i+I)/2][(j+J)/2], Vertices[(i+I)/2][J], c,
                Vertices[I][(j+J)/2]);
     DrawSquare(Vertices[(i+I)/2][j], Vertices[(i+I)/2][(j+J)/2],
                Vertices[I][(j+J)/2], d);
   }

   /* Recursive call the current function to divide the square further ...
      ... decrease the number of the subdivsion (not the global number S)
      by 1 */
   DivideSquare(a, Vertices[i][(j+J)/2], Vertices[(i+I)/2][(j+J)/2],
                Vertices[(i+I)/2][j], i, j, (i+I)/2, (j+J)/2, S-1);
   DivideSquare(Vertices[i][(j+J)/2], b, Vertices[(i+I)/2][J],
                Vertices[(i+I)/2][(j+J)/2], i, (j+J)/2, (i+I)/2, J, S-1);
   DivideSquare(Vertices[(i+I)/2][(j+J)/2], Vertices[(i+I)/2][J], c,
                Vertices[I][(j+J)/2], (i+I)/2, (j+J)/2, I, J, S-1);
   DivideSquare(Vertices[(i+I)/2][j], Vertices[(i+I)/2][(j+J)/2],
                Vertices[I][(j+J)/2], d, (i+I)/2, j, I, (j+J)/2, S-1);
                     
  } 
}     

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    display                                      +
   +                                                            +
   +  Purpose: To display the lines, clipping box and the       +
   +           clipped lines.                                   +
   +                                                            +
   +  Inputs: none                                              +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void display(void)
{
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   SetCameraPosition();
      
   DivideSquare(Vertices[0][0], Vertices[0][64], Vertices[64][64],
                Vertices[64][0], 0, 0, 64, 64, S);
                    
   Calc = 0;  /* set the status to avoid calculations next time */
    
   glutSwapBuffers();
}

/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    KeyPress                                     +
   +                                                            +
   +  Purpose: This function will check to see if the key       +
   +            pressed is a q (or Q). If so, then the program  +
   +            will terminate gracefully.                      +
   +                                                            +
   +  Inputs: Key (unsigned char) - Key pressed.                +
   +          X (int) - x location of mouse pointer.            +
   +          Y (int) - y location of mouse pointer.            +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void KeyPress(unsigned char Key, int X, int Y)
{
  GLint Old_S = S;    /* Assign the current subdivision 
                         value to a temp variable. */

  if (toupper(Key) == 'Q')
    exit(0);

  /* Determine if a subdivision was selected by the user ... if
     he/she press the keys between 1 and 6 (inclusive) then 
     assign the corresponding subdivion and perform the calculations */
  switch(Key)
  {
    case '1':
         S = 1;  break;
    case '2':
         S = 2;  break;
    case '3':
         S = 3;  break;
    case '4':
         S = 4;  break;
    case '5':
         S = 5;  break;
    case '6':
         S = 6;  break;
    default:
      break;
  } 

  /* If the new subdivion is different from the old one then 
     recalculate and redraw everything otherwise don't do anything. */
  if ( S != Old_S )
  {
    Calc = 1;
    InitVertices();
    glutPostRedisplay();
  }
}
 
 
/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    MouseButtonPressed                           +
   +                                                            +
   +  Purpose: To capture the input of endpoint for a line.     +
   +           On the depression of the left mouse button the   +
   +           current cursor position is used to capture the   +
   +           coordinates at the current point which are used  +
   +           to define the first or second endpoint.          +
   +                                                            +
   +  Inputs: Button - determines which button is selected.     +
   +          State - determines if button is pressed/released  +
   +          X,Y - coordinates of current cursor               +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
void MouseButtonPressed(int Button, int State, int X, int Y)
{
  static int LastX = 0;  /* Holds the old X coord of mouse */
  static int LastY = 0;  /* Holds the old Y coord of mouse */  

  if (State == GLUT_DOWN)
    if (Button == GLUT_LEFT_BUTTON)
    {
      if (X > LastX) 
        Azimuth = (Azimuth - 3) % 360;
      else if (X < LastX) 
        Azimuth = (Azimuth + 3) % 360;

      if (Y > LastY) 
        Elevation = (Elevation + 3) % 360;
      else if (Y < LastY) 
        Elevation = (Elevation - 3) % 360; 
    }
    else if (Button == GLUT_RIGHT_BUTTON)
    {
      if (X > LastX) 
        Twist = (Twist + 3) % 360;
      else if (X < LastX) 
        Twist = (Twist - 3) % 360;
   
      if ((Y > LastY) && (Focal > -16.0))
        Focal -= 0.5;
      else if ((Y < LastY) && (Focal < -6.0))
        Focal += 0.5;
    }

  LastX = X;
  LastY = Y;

  glutPostRedisplay();
}


/* ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   +                                                            +
   +  Function:    main                                         +
   +                                                            +
   +  Purpose: This is the main function of mp5. Most of the    +
   +            initialization of the display is done here and  +
   +            functions for events are invoked here.          +
   +                                                            +
   +  Inputs: none.                                             +
   +                                                            +
   ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ */
int main(int argc, char **argv)
{
  glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);

  /* Display quit procedure. */
  printf("CS318 - MP5 - Fractal Mountain\n");
  printf("INPUTS:\n");
  printf("Left Mouse Button  - Click/Hold to Rotate Camera Azimuth/Elevation\n");
  printf("Right Mouse Button - Click/Hold to Zoom (Up/Down) or Twist (Left/Right)\n");
  printf("'1' to '6'         - Set Level of Detail / Subdivision\n");
  printf("'q' or 'Q'         - Quit Program\n");

  /* Initialize window. */
  glutInit(&argc, argv);
  glutInitWindowSize(WinWidth, WinWidth);
  glutInitWindowPosition(100, 100);
  glutCreateWindow("MP5: Fractal Mountain");
  init();
 
  /* Draw the S=1 mountain */
  InitVertices();

  /* Define camera position. */
  SetCameraPosition();

  /* Call display function and enter main loop. */
  glutDisplayFunc(display);
  glutMouseFunc(MouseButtonPressed);
  glutKeyboardFunc(KeyPress);
  glutReshapeFunc(reshape);
  glutMainLoop();

  return 0;
}

